require("dotenv").config();
const express = require("express");
const mongoose = require("mongoose");
const cors = require("cors");
const bodyParser = require("body-parser");

const Image = require("./models/imageModel");

const app = express();
app.use(cors());
app.use(bodyParser.json({ limit: "10mb" })); // Increase body size limit for Base64 images

// Connect to MongoDB
mongoose.connect(process.env.MONGO_URI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
}).then(() => console.log("MongoDB Connected"))
    .catch(err => console.error(err));

// Upload Image (Base64)
app.post("/api/upload", async (req, res) => {
    try {
        const { base64 } = req.body; // Expecting { base64: "data:image/png;base64,..." }

        if (!base64) {
            return res.status(400).json({ error: "No image data provided" });
        }

        const newImage = new Image({ base64 });
        await newImage.save();
        res.status(201).json({ message: "Image uploaded successfully", image: newImage });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.post("/api/upload-multiple", async (req, res) => {
    try {
        const { base64Images } = req.body; // Expecting { base64Images: ["data:image/png;base64,...", ...] }

        if (!base64Images || !Array.isArray(base64Images) || base64Images.length === 0) {
            return res.status(400).json({ error: "No images provided" });
        }

        const imageCount = await Image.countDocuments();
        const maxImages = 50;
        const newImagesCount = base64Images.length;

        // If adding new images will exceed the limit, delete the oldest X images
        if (imageCount + newImagesCount > maxImages) {
            const deleteCount = (imageCount + newImagesCount) - maxImages;
            const oldestImages = await Image.find().sort({ createdAt: 1 }).limit(deleteCount); // Oldest images

            const deleteIds = oldestImages.map(img => img._id);
            await Image.deleteMany({ _id: { $in: deleteIds } });
        }

        // Save new images
        const imageDocs = base64Images.map(base64 => ({ base64 }));
        const savedImages = await Image.insertMany(imageDocs);

        res.status(201).json({ message: "Images uploaded successfully", images: savedImages });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});



// Get All Images
app.get("/api/images", async (req, res) => {
    try {
        const images = await Image.find().sort({ createdAt: -1 });
        res.status(200).json(images);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// GET latest 6 images
app.get('/api/latest', async (req, res) => {
    try {
        const latestImages = await Image.find().sort({ createdAt: -1 }).limit(6); // Fetch 6 latest images
        res.status(200).json(latestImages);
    } catch (error) {
        res.status(500).json({ message: 'Error fetching images', error });
    }
});

const PORT = process.env.PORT || 5000;
app.listen(PORT, () => console.log(`Server running on port ${PORT}`));
